<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Banner Helper
 * 
 * This helper provides functions to display dynamic banners on different pages
 * 
 * USAGE EXAMPLES:
 * 
 * 1. For Home Page (Slider):
 *    $this->load->helper('banner');
 *    echo display_page_banner('home', 'Default Title', true);
 * 
 * 2. For Other Pages (Single Banner):
 *    $this->load->helper('banner');
 *    echo display_page_banner('shop', 'Shop Page Title', false);
 * 
 * 3. Get Banner Data Only:
 *    $banner = get_page_banner('courses');
 *    if ($banner) {
 *        echo $banner['img_text'];
 *    }
 * 
 * ADMIN SETUP:
 * 1. Go to Admin Panel > Slider Management
 * 2. Select the page where you want the banner to appear
 * 3. Upload image and set sort order
 * 4. Optionally add banner text and link
 * 5. Save - banner will appear on selected page
 */

if (!function_exists('display_page_banner')) {
    /**
     * Display dynamic banner for a specific page
     * 
     * @param string $page_name - The page identifier (home, shop, courses, etc.)
     * @param string $fallback_title - Fallback title if no banner text is set
     * @param bool $show_slider - Whether to show as slider or single banner
     * @return string HTML output
     */
    function display_page_banner($page_name, $fallback_title = '', $show_slider = false) {
        $CI =& get_instance();
        
        // Fetch banners for specific page
        $query = $CI->db->query("SELECT * FROM yn_site_img WHERE img_place='slider1' AND img_ex=? AND img_status='1' ORDER BY img_sort ASC LIMIT 5", [$page_name]);
        $banners = $query->result_array();
        
        if (empty($banners)) {
            return '';
        }
        
        $html = '';
        
        if ($show_slider) {
            // Display as slider - use exact same structure as home page
            $html .= '<section class="hero-slider">';
            foreach ($banners as $index => $banner) {
                $active_class = $index === 0 ? 'active' : '';
                $html .= '<div class="slide ' . $active_class . '" style="background-image:url(\'' . base_url('assets/avator/upload/' . $banner['img_name']) . '\');">';
                $html .= '<div class="overlay"></div>';
                $html .= '<div class="content">';
                $html .= '<div class="content-box">';
                $html .= '<h1>' . (!empty($banner['img_text']) ? htmlspecialchars($banner['img_text']) : $fallback_title) . '</h1>';
                $html .= '<p><i class="fas fa-lightbulb"></i> Books for students, kids, and professionals</p>';
                $html .= '<a href="' . (!empty($banner['img_link']) ? $banner['img_link'] : base_url('shop')) . '" class="btn">Shop Now <i class="fas fa-arrow-right"></i></a>';
                $html .= '</div>';
                $html .= '</div>';
                $html .= '</div>';
            }
            $html .= '<div class="dots"></div>';
            $html .= '</section>';
        } else {
            // Display as single banner with proper CSS
            $banner = $banners[0]; // Use first banner
            $html .= '<style>';
            $html .= '.banner-wrapper {';
            $html .= 'background-image: url(\'' . base_url('assets/avator/upload/' . $banner['img_name']) . '\');';
            $html .= 'height: 250px;';
            $html .= 'background-size: cover;';
            $html .= 'background-position: center;';
            $html .= 'display: flex;';
            $html .= 'align-items: center;';
            $html .= 'justify-content: center;';
            $html .= 'text-align: center;';
            $html .= '}';
            $html .= '.banner-title {';
            $html .= 'color: #0A8FD6;';
            $html .= 'display: inline-block;';
            $html .= 'background: #E8F6FE;';
            $html .= 'padding: 15px 25px;';
            $html .= 'border-radius: 8px;';
            $html .= 'font-size: 2rem;';
            $html .= 'max-width: 90%;';
            $html .= 'word-wrap: break-word;';
            $html .= '}';
            $html .= '</style>';
            $html .= '<div class="banner-wrapper">';
            $html .= '<h1 class="banner-title">' . (!empty($banner['img_text']) ? htmlspecialchars($banner['img_text']) : $fallback_title) . '</h1>';
            $html .= '</div>';
        }
        
        return $html;
    }
}

if (!function_exists('get_page_banner')) {
    /**
     * Get banner data for a specific page
     * 
     * @param string $page_name - The page identifier
     * @return array Banner data
     */
    function get_page_banner($page_name) {
        $CI =& get_instance();
        $query = $CI->db->query("SELECT * FROM yn_site_img WHERE img_place='slider1' AND img_ex=? AND img_status='1' ORDER BY img_sort ASC LIMIT 1", [$page_name]);
        return $query->row_array();
    }
}
